<?php
/**
 * Exemplo de teste de integração com a API PIX
 * 
 * Este arquivo demonstra como usar a API do velozpay
 * para criar e consultar pagamentos.
 */

// Configuração da API
$api_url = 'https://velozpay.online/api/v1/gateway/processa_pagamento_api.php';
$api_key = 'SUA_API_KEY_AQUI'; // Substitua pela sua API key

/**
 * Exemplo 1: Criar um pagamento PIX
 */
function criarPagamentoPIX($api_url, $api_key) {
    echo "<h2>🔄 Criando Pagamento PIX</h2>\n";
    
    $dados_pagamento = array(
        'api-key' => $api_key,
        'name' => 'João Silva Teste',
        'cpf' => '123.456.789-00',
        'amount' => 25.50,
        'tipoPagamento' => 'PIX',
        'email' => 'joao@teste.com',
        'telefone' => '(11) 99999-9999'
    );
    
    echo "<strong>Dados enviados:</strong><br>\n";
    echo "<pre>" . json_encode($dados_pagamento, JSON_PRETTY_PRINT) . "</pre>\n";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($dados_pagamento));
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'Content-Type: application/json',
        'Accept: application/json'
    ));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);
    
    echo "<strong>Status HTTP:</strong> $http_code<br>\n";
    
    if ($curl_error) {
        echo "<span style='color: red;'>❌ Erro cURL: $curl_error</span><br>\n";
        return null;
    }
    
    echo "<strong>Resposta da API:</strong><br>\n";
    echo "<pre>" . json_encode(json_decode($response, true), JSON_PRETTY_PRINT) . "</pre>\n";
    
    $data = json_decode($response, true);
    
    if ($http_code === 200 && isset($data['status']) && $data['status'] === 'success') {
        echo "<span style='color: green;'>✅ Pagamento criado com sucesso!</span><br>\n";
        echo "<strong>ID da Transação:</strong> " . $data['idTransaction'] . "<br>\n";
        
        if (isset($data['paymentCode'])) {
            echo "<strong>Código PIX:</strong><br>\n";
            echo "<textarea readonly style='width: 100%; height: 60px;'>" . $data['paymentCode'] . "</textarea><br>\n";
        }
        
        if (isset($data['qrCodeImage'])) {
            echo "<strong>QR Code:</strong><br>\n";
            echo "<img src='" . $data['qrCodeImage'] . "' alt='QR Code PIX' style='max-width: 200px;'><br>\n";
        }
        
        return $data['idTransaction'];
    } else {
        echo "<span style='color: red;'>❌ Erro ao criar pagamento</span><br>\n";
        return null;
    }
}

/**
 * Exemplo 2: Consultar status de um pagamento
 */
function consultarStatusPagamento($api_url, $api_key, $id_transacao) {
    echo "<h2>🔍 Consultando Status do Pagamento</h2>\n";
    
    if (!$id_transacao) {
        echo "<span style='color: orange;'>⚠️ ID da transação não fornecido</span><br>\n";
        return;
    }
    
    $url_consulta = $api_url . '?api-key=' . urlencode($api_key) . '&idTransaction=' . urlencode($id_transacao);
    
    echo "<strong>URL de consulta:</strong> $url_consulta<br>\n";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url_consulta);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'Accept: application/json'
    ));
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);
    
    echo "<strong>Status HTTP:</strong> $http_code<br>\n";
    
    if ($curl_error) {
        echo "<span style='color: red;'>❌ Erro cURL: $curl_error</span><br>\n";
        return;
    }
    
    echo "<strong>Resposta da API:</strong><br>\n";
    echo "<pre>" . json_encode(json_decode($response, true), JSON_PRETTY_PRINT) . "</pre>\n";
    
    $data = json_decode($response, true);
    
    if ($http_code === 200 && isset($data['status']) && $data['status'] === 'success') {
        echo "<span style='color: green;'>✅ Consulta realizada com sucesso!</span><br>\n";
        
        if (isset($data['pedido'])) {
            $pedido = $data['pedido'];
            echo "<strong>Status do Pedido:</strong> " . $pedido['status'] . "<br>\n";
            echo "<strong>Valor:</strong> R$ " . number_format($pedido['amount'], 2, ',', '.') . "<br>\n";
            echo "<strong>Cliente:</strong> " . $pedido['clientName'] . "<br>\n";
            echo "<strong>Data:</strong> " . $pedido['realData'] . "<br>\n";
        }
        
        if (isset($data['asaas'])) {
            $asaas = $data['asaas'];
            echo "<strong>Status Asaas:</strong> " . $asaas['status'] . "<br>\n";
            echo "<strong>Data de Vencimento:</strong> " . $asaas['dueDate'] . "<br>\n";
            if ($asaas['paymentDate']) {
                echo "<strong>Data de Pagamento:</strong> " . $asaas['paymentDate'] . "<br>\n";
            }
        }
    } else {
        echo "<span style='color: red;'>❌ Erro ao consultar status</span><br>\n";
    }
}

/**
 * Exemplo 3: Testar conectividade básica
 */
function testarConectividade($api_url) {
    echo "<h2>🌐 Testando Conectividade</h2>\n";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_NOBODY, true);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);
    
    echo "<strong>URL testada:</strong> $api_url<br>\n";
    echo "<strong>Status HTTP:</strong> $http_code<br>\n";
    
    if ($curl_error) {
        echo "<span style='color: red;'>❌ Erro de conectividade: $curl_error</span><br>\n";
    } elseif ($http_code >= 200 && $http_code < 300) {
        echo "<span style='color: green;'>✅ Conectividade OK</span><br>\n";
    } elseif ($http_code >= 400) {
        echo "<span style='color: orange;'>⚠️ Servidor respondeu com código $http_code</span><br>\n";
    } else {
        echo "<span style='color: red;'>❌ Resposta inesperada: $http_code</span><br>\n";
    }
}

/**
 * Executar testes
 */
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teste de Integração - velozpay</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            max-width: 800px;
            margin: 20px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            border-bottom: 2px solid #3498db;
            padding-bottom: 10px;
        }
        h2 {
            color: #34495e;
            margin-top: 30px;
        }
        pre {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 4px;
            border-left: 4px solid #3498db;
            overflow-x: auto;
        }
        textarea {
            font-family: 'Courier New', monospace;
            font-size: 12px;
        }
        .warning {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            color: #856404;
            padding: 15px;
            border-radius: 4px;
            margin: 20px 0;
        }
        .info {
            background: #d1ecf1;
            border: 1px solid #bee5eb;
            color: #0c5460;
            padding: 15px;
            border-radius: 4px;
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔌 Teste de Integração - velozpay</h1>
        
        <div class="warning">
            <strong>⚠️ Importante:</strong> Substitua a variável <code>$api_key</code> no início deste arquivo pela sua API key real antes de executar os testes.
        </div>
        
        <div class="info">
            <strong>ℹ️ Informação:</strong> Este script testa a integração com a API do velozpay. Use apenas em ambiente de desenvolvimento/teste.
        </div>
        
        <?php
        // Verificar se a API key foi configurada
        if ($api_key === 'SUA_API_KEY_AQUI') {
            echo '<div style="background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; border-radius: 4px; margin: 20px 0;">';
            echo '<strong>❌ Erro:</strong> Configure sua API key antes de executar os testes.';
            echo '</div>';
        } else {
            // Executar testes
            testarConectividade($api_url);
            echo "<hr>\n";
            
            $id_transacao = criarPagamentoPIX($api_url, $api_key);
            echo "<hr>\n";
            
            if ($id_transacao) {
                consultarStatusPagamento($api_url, $api_key, $id_transacao);
            }
        }
        ?>
        
        <h2>📋 Sobre este Teste</h2>
        <p>Este script demonstra como integrar com a API do velozpay:</p>
        <ul>
            <li><strong>Conectividade:</strong> Testa se a API está acessível</li>
            <li><strong>Criar Pagamento:</strong> Envia uma requisição POST para criar um pagamento PIX</li>
            <li><strong>Consultar Status:</strong> Faz uma consulta GET para verificar o status do pagamento</li>
        </ul>
        
        <h2>🔧 Como Usar</h2>
        <ol>
            <li>Configure sua API key na variável <code>$api_key</code></li>
            <li>Execute este arquivo em um servidor PHP</li>
            <li>Verifique os resultados dos testes</li>
            <li>Use este código como base para sua integração</li>
        </ol>
        
        <h2>📚 Documentação</h2>
        <p>Para mais informações sobre a API, consulte o arquivo <code>README_API.md</code> na pasta principal do projeto.</p>
    </div>
</body>
</html>
